unit xTimer;

(**************************************************************)
(** //               XENGINE Timer Unit                   // **)
(** //  (C) 2025 Coded by Adam Kozinski & Dominik Galoch  // **)
(** //////////////////////////////////////////////////////// **)
(**************************************************************)

interface

uses dos;

var
    OldInt1C    : pointer;          { Stores the original interrupt vector }
    timer_count : longint;          { Time counter in milliseconds }

{//////////////////////////////////////////////////////////}
{ PROCEDURE AND FUNCTION HEADERS }
{//////////////////////////////////////////////////////////}

procedure xInitTimer(freq : word);    { Initializes timer with given frequency in Hz }
function  xGetTime : longint;         { Returns current time in milliseconds }
procedure xWait(ms : word);           { Waits for the specified number of milliseconds }
procedure xRestoreTimer;              { Restores original timer settings }

{//////////////////////////////////////////////////////////}

implementation

(***********************************************************)

procedure NewInt1C; interrupt;
begin
    Inc(timer_count);   { Increments the counter on every interrupt }
    asm
        mov al, 20h     { Sends EOI signal to the interrupt controller }
        out 20h, al
    end;
end;

(***********************************************************)

procedure xInitTimer(freq : word);
var
    count : word;
begin
    if freq = 0 then freq := 1000;      { Default frequency: 1000 Hz }
    count := 1193180 div freq;          { Calculates PIT counter value }

    asm
        cli                             { Disable interrupts }
        mov al, 36h                     { Sets PIT mode (channel 0, mode 2) }
        out 43h, al
        mov ax, count                   { Send low and high byte }
        out 40h, al
        mov al, ah
        out 40h, al
        sti                             { Enable interrupts }
    end;

    GetIntVec($1C, OldInt1C);           { Save original interrupt vector }
    SetIntVec($1C, @NewInt1C);          { Set new interrupt handler }
    timer_count := 0;                   { Reset counter }
end;

(***********************************************************)

function xGetTime : longint;
begin
    xGetTime := timer_count;            { Returns current time }
end;

(***********************************************************)

procedure xWait(ms : word);
var
    start : LongInt;
begin
    start := xGetTime;                  { Get starting time }
    while xGetTime - start < ms do;     { Wait until desired time passes }
end;

(***********************************************************)

procedure xRestoreTimer;
begin
    SetIntVec($1C, OldInt1C);           { Restore original interrupt vector }
    asm
        cli
        mov al, 36h                     { Restore default PIT settings }
        out 43h, al
        mov ax, 0                       { 65536 (default ~18.2 Hz) }
        out 40h, al
        mov al, ah
        out 40h, al
        sti
    end;
end;

(***********************************************************)

begin
    xInitTimer(1000);  { Initializes timer with 1000 Hz frequency }
    { Note: Call xRestoreTimer procedure before program termination! }
end.